package jingdong

import (
	"context"
	"github.com/gogf/gf/encoding/gjson"
	"github.com/gogf/gf/frame/g"
	"github.com/gogf/gf/util/gconv"
)

type refundJD struct {
}

var Refund = refundJD{}

type RefundCancelRes struct {
	CommonRes
	Result bool `json:"result"`
}

type ApplyReq struct {
	JdOrderId             int64                   `json:"jdOrderId"`             //京东订单号
	CustomerExpect        uint                    `json:"customerExpect"`        //售后类型：退货(10)、换货(20)、维修(30)
	QuestionDesc          string                  `json:"questionDesc"`          //产品问题描述，最多1000字符
	IsNeedDetectionReport bool                    `json:"isNeedDetectionReport"` //是否需要检测报告
	IsHasPackage          bool                    `json:"isHasPackage"`          //是否有包装
	QuestionPic           string                  `json:"questionPic"`           //问题描述图片.最多2000字符
	PackageDesc           int                     `json:"packageDesc"`           //包装描述：0 无包装	10 包装完整20 包装破损
	UserInfoDto           *AfterSaleCustomerDto   `json:"asCustomerDto"`         //客户信息实体
	PickDto               *AfterSalePickwareDto   `json:"asPickwareDto"`         //取件信息实体
	ReturnDto             *AfterSaleReturnwareDto `json:"asReturnwareDto"`       //返件信息实体
	DetailDto             *AfterSaleDetailDto     `json:"asDetailDto"`           //申请单明细
}

type AfterSaleCustomerDto struct {
	Name  string `json:"customerContactName"` //联系人，最多50字符，
	Tel   string `json:"customerTel"`         //联系电话，最多50字符
	Phone string `json:"customerMobilePhone"` //手机号，最多50字符
}

type AfterSalePickwareDto struct {
	Type     int    `json:"pickwareType"`     //取件方式：4上门取件； 7 客户送货；40客户发货
	Province int    `json:"pickwareProvince"` //取件省，必填
	City     int    `json:"pickwareCity"`     //取件市，必填
	County   int    `json:"pickwareCounty"`   //取件县，必填
	Village  int    `json:"pickwareVillage"`  //取件乡镇
	Address  string `json:"pickwareAddress"`  //取件街道地址，取件方式为4（上门取件）时必填，最多500字符
}

type AfterSaleReturnwareDto struct {
	Type     int    `json:"returnwareType"`     //返件方式：自营配送(10),第三方配送(20);
	Province int    `json:"returnwareProvince"` //返件省
	City     int    `json:"returnwareCity"`     //返件市
	County   int    `json:"returnwareCounty"`   //返件县
	Village  int    `json:"returnwareVillage"`  //返件乡镇
	Address  string `json:"returnwareAddress"`  //返件街道地址，最多500字符
}

type AfterSaleDetailDto struct {
	Id  int64 `json:"skuId"`  //商品编号
	Num uint  `json:"skuNum"` //商品申请数量
}

type GetServiceListReq struct {
	JdOrderID int64 `json:"jdOrderId"` //京东订单号
	PageIndex int   `json:"pageIndex"` //页码，1代表第一页
	PageSize  int   `json:"pageSize"`  //每页记录数, 大小取值范围[1,100]
}

type RefundListRes struct {
	CommonRes
	Result struct {
		TotalNum        int `json:"totalNum"`  //总记录数
		PageSize        int `json:"pageSize"`  //每页记录数
		PageNum         int `json:"pageNum"`   //总页数
		PageIndex       int `json:"pageIndex"` //当前页数
		ServiceInfoList []struct {
			AfsServiceId       int64  `json:"afsServiceId"` //服务单号
			CustomerExpect     int    `json:"customerExpect"`
			CustomerExpectName string `json:"customerExpectName"`
			AfsApplyTime       string `json:"afsApplyTime"`
			OrderId            int64  `json:"orderId"`
			WareId             int64  `json:"wareId"` //商品编号
			WareName           string `json:"wareName"`
			AfsServiceStep     int    `json:"afsServiceStep"`
			AfsServiceStepName string `json:"afsServiceStepName"`
			Cancel             int    `json:"cancel"` //是否可取消: 0代表否，1代表是
		} `json:"serviceInfoList"`
	} `json:"result"`
}

type RefundDetailRes struct {
	CommonRes
	Result struct {
		AfsServiceId                    int64       `json:"afsServiceId"`   //京东售后服务单号
		CustomerExpect                  int         `json:"customerExpect"` //服务类型码：退货(10)、换货(20)、维修(20)
		AfsApplyTime                    string      `json:"afsApplyTime"`   //服务单申请时间：格式为yyyy-MM-dd HH:mm:ss
		OrderId                         int64       `json:"orderId"`        //京东订单号
		IsHasInvoice                    int         `json:"isHasInvoice"`
		IsNeedDetectionReport           int         `json:"isNeedDetectionReport"`
		IsHasPackage                    int         `json:"isHasPackage"`
		QuestionPic                     string      `json:"questionPic"`
		QuestionDesc                    string      `json:"questionDesc"`
		ApproveNotes                    string      `json:"approveNotes"`   //审核意见
		ApprovedResult                  int         `json:"approvedResult"` //审核结果：直赔积分 (11),直赔余额 (12),直赔优惠卷 (13),直赔京豆 (14),直赔商品 (21),上门换新 (22),自营取件 (31),客户送货(32),客户发货 (33),闪电退款 (34),虚拟退款 (35),大家电检测 (80),大家电安装 (81),大家电移机 (82),大家电维修 (83),大家电其它(84);
		ApprovedResultName              string      `json:"approvedResultName"`
		AfsServiceStep                  int         `json:"afsServiceStep"` //服务单环节：申请阶段(10),审核不通过(20),客服审核(21),商家审核(22),京东收货(31),商家收货(32), 京东处理(33),商家处理(34), 用户确认(40),完成(50), 取消(60);
		AfsServiceStepName              string      `json:"afsServiceStepName"`
		ProcessResult                   int         `json:"processResult"` //处理结果：返修换新(23),退货(40), 换良(50),原返(60),病单(71),出检(72),维修(73),强制关单 (80),线下换新(90)
		ProcessResultName               interface{} `json:"processResultName"`
		ServiceAftersalesAddressInfoDTO struct {
			Address  string `json:"address"`  //售后地址
			Tel      string `json:"tel"`      //售后电话
			LinkMan  string `json:"linkMan"`  //售后联系人
			PostCode string `json:"postCode"` //售后邮编
		} `json:"serviceAftersalesAddressInfoDto"` //售后地址信息
		ServiceFinanceDetailInfoDTOs []struct {
			ServiceFinanceDetailInfoDTO
			RefundPrice float64 `json:"refundPrice"` //退款金额
		} `json:"serviceFinanceDetailInfoDTOs"` //退款明细
		ServiceTrackInfoDTOs []struct {
			AfsServiceId int64  `json:"afsServiceId"`
			Title        string `json:"title"`
			Context      string `json:"context"`
			CreateDate   string `json:"createDate"`
			CreateName   string `json:"createName"`
			CreatePin    string `json:"createPin"`
		} `json:"serviceTrackInfoDTOs"`
		ServiceDetailInfoDTOs []struct {
			WareId        int64  `json:"wareId"`
			WareName      string `json:"wareName"`
			WareBrand     string `json:"wareBrand"`
			AfsDetailType int    `json:"afsDetailType"`
		} `json:"serviceDetailInfoDTOs"`
	} `json:"result"`
}

type ServiceFinanceDetailInfoDTO struct {
	RefundWay     int    `json:"refundWay"`     //退款方式
	RefundWayName string `json:"refundWayName"` //退款方式名称
	Status        int    `json:"status"`        //状态
	StatusName    string `json:"statusName"`    //状态名称
	WareName      string `json:"wareName"`      //商品名称
	WareId        int64  `json:"wareId"`        //商品编号
}

type RefundCommonReq struct {
	OrderID int64 `json:"jdOrderId"` //订单编号
	SkuID   int64 `json:"skuId"`     //商品编号
}

type RefundIsSupportRes struct {
	CommonRes
	Result int `json:"result"` //可申请时返回可申请数量
}

type RefundSupportTypeRes struct {
	CommonRes
	Result []*CodeItem `json:"result"` //服务类型码：退货(10)、换货(20)、维修(30) 服务类型名称：退货、换货、维修
}

type CodeItem struct {
	Code string `json:"code"` //编码
	Name string `json:"name"` //名称
}

type RefundReturnTypeRes struct {
	CommonRes
	Result []*CodeItem `json:"result"` //服务类型码：上门取件(4)、客户发货(40)、客户送货(7)  服务类型名称：上门取件、客户发货、客户送货
}

type RefundSendReq struct {
	AfsServiceId   int64  `json:"afsServiceId"`
	DeliverDate    string `json:"deliverDate"`
	ExpressCode    string `json:"expressCode"`
	ExpressCompany string `json:"expressCompany"`
	FreightMoney   int    `json:"freightMoney"`
}

// Send 根据填写单号
func (*refundJD) Send(ctx context.Context, req *RefundSendReq) (res *CommonRes, err error) {
	method := "afterSale/updateSendSku"
	param := g.Map{
		"param": gjson.New(req).MustToJsonString(),
	}
	result, err := server.requestApi(ctx, method, param)
	if err != nil {
		return
	}
	_ = gjson.New(result).Scan(&res)
	return
}

// Cancel 取消售后
func (*refundJD) Cancel(ctx context.Context, id interface{}) (res *RefundCancelRes, err error) {
	method := "afterSale/auditCancel"
	req := g.Map{
		"serviceIdList": gconv.Ints(id),
		"approveNotes":  "取消售后",
	}
	param := g.Map{
		"param": gjson.New(req).MustToJsonString(),
	}
	result, err := server.requestApi(ctx, method, param)
	if err != nil {
		return
	}
	_ = gjson.New(result).Scan(&res)
	return
}

// Apply 申请售后服务（退货、换货、维修）
func (*refundJD) Apply(ctx context.Context, req *ApplyReq) (res *CommonRes, err error) {
	method := "afterSale/createAfsApply"
	param := g.Map{
		"param": gjson.New(req).MustToJsonString(),
	}
	result, err := server.requestApi(ctx, method, param)
	if err != nil {
		return
	}
	_ = gjson.New(result).Scan(&res)
	return
}

// List 查询服务单根据客户账号和订单号分页查询服务单概要信息
func (*refundJD) List(ctx context.Context, OrderSn string) (res *RefundListRes, err error) {
	method := "afterSale/getServiceListPage"
	param := g.Map{
		"param": gjson.New(&GetServiceListReq{
			JdOrderID: gconv.Int64(OrderSn),
			PageIndex: 1,
			PageSize:  100,
		}).MustToJsonString(),
	}
	result, err := server.requestApi(ctx, method, param)
	if err != nil {
		return
	}
	_ = gjson.New(result).Scan(&res)
	return
}

// Detail 根据服务单号查询服务单明细信息
func (*refundJD) Detail(ctx context.Context, req string) (res *RefundDetailRes, err error) {
	method := "afterSale/getServiceDetailInfo"
	var reqJson = gjson.New("")
	_ = reqJson.Set("afsServiceId", gconv.Int64(req))
	_ = reqJson.Set("appendInfoSteps", []int{1, 4, 5}) //1、代表增加获取售后地址信息；	2、代表增加获取客户发货信息；	3、代表增加获取退款明细；	4、代表增加获取跟踪信息；	5、代表增加获取允许的操作信息

	param := g.Map{
		"param": reqJson.MustToJsonString(),
	}
	result, err := server.requestApi(ctx, method, param)
	if err != nil {
		return
	}
	_ = gjson.New(result).Scan(&res)
	return
}

// IsSupport 查询订单商品是否允许售后
func (*refundJD) IsSupport(ctx context.Context, req *RefundCommonReq) (res *RefundIsSupportRes, err error) {
	method := "afterSale/getAvailableNumberComp"
	param := g.Map{
		"param": gjson.New(req).MustToJsonString(),
	}
	result, err := server.requestApi(ctx, method, param)
	if err != nil {
		return
	}
	_ = gjson.New(result).Scan(&res)
	return
}

// SupportType 查询订单商品售后类型
func (*refundJD) SupportType(ctx context.Context, req *RefundCommonReq) (res *RefundSupportTypeRes, err error) {
	method := "afterSale/getCustomerExpectComp"
	param := g.Map{
		"param": gjson.New(req).MustToJsonString(),
	}
	result, err := server.requestApi(ctx, method, param)
	if err != nil {
		return
	}

	_ = gjson.New(result).Scan(&res)
	return
}

// ReturnType 查询支持的商品返回京东方式
func (*refundJD) ReturnType(ctx context.Context, req *RefundCommonReq) (res *RefundReturnTypeRes, err error) {
	method := "afterSale/getWareReturnJdComp"
	param := g.Map{
		"param": gjson.New(req).MustToJsonString(),
	}
	result, err := server.requestApi(ctx, method, param)
	if err != nil {
		return
	}
	_ = gjson.New(result).Scan(&res)
	return
}
