package jingdong

import (
	"context"
	"github.com/gogf/gf/encoding/gjson"
	"github.com/gogf/gf/frame/g"
)

type orderJD struct {
}

//Order 订单
var Order = orderJD{}

//PriceSnap 商品价格
type PriceSnap struct {
	SkuID int     `json:"skuId"` //商品编号
	Price float64 `json:"price"` //商品价格  格式:21.30
}

//OrderCreateReq 下单
type OrderCreateReq struct {
	OrderSn     string //订单单号
	SkuNums     []*SkuNums
	Consignee   string //收货人姓名，最多20个字符
	Phone       string //手机号，最多20个字符
	Address     *PlaceOrderAddress
	PriceSnap   []*PriceSnap
	InvoiceType InvoiceType
}

//PlaceOrderAddress 下单地址
type PlaceOrderAddress struct {
	Province string //一级地址编码
	City     string //二级地址编码
	County   string //三级地址编码
	Town     string //四级地址编码
	Desc     string //收货人详细地址，最多100个字符
}

//OrderCreateRes 下单
type OrderCreateRes struct {
	*CommonRes
	Result struct {
		JdOrderID       int64           `json:"jdOrderId"`       //京东订单号
		OrderPrice      float64         `json:"orderPrice"`      //订单总金额
		OrderNakedPrice float64         `json:"orderNakedPrice"` //订单未税金额
		OrderTaxPrice   float64         `json:"orderTaxPrice"`   //订单税额
		Sku             []*OrderSkuItem `json:"sku"`
	} `json:"result"`
}

//OrderSkuItem 订单商品
type OrderSkuItem struct {
	SkuID      int     `json:"skuId"`      //商品编号
	Num        int     `json:"num"`        //商品数量
	Price      float64 `json:"price"`      //商品单价
	Name       string  `json:"name"`       //商品名称
	NakedPrice float64 `json:"nakedPrice"` //商品未税价
}

//OrderReflectRes 反查订单
type OrderReflectRes struct {
	*CommonRes
	Result string `json:"result"` //第三方订单号（非京东订单号）
}

type OrderDetailBase struct {
	*CommonRes
	Result struct {
		JdOrderID int64 `json:"jdOrderId"` //京东订单编号
		Type      int   `json:"type"`      //订单类型。1是父订单   2是子订单
	}
}

//OrderDetailRes 订单详情
// jdOrderState 京东状态
// 1.新单 2.等待支付 3.等待支付确认 4.延迟付款确认 5.订单暂停 6.店长最终审核 7.等待打印 8.等待出库 9.等待打包 10.等待发货
// 11.自提途中 12.上门提货 13.自提退货 14.确认自提 16.等待确认收货 17.配送退货 18.货到付款确认 19.已完成 21.收款确认 22.锁定
type OrderDetailRes struct {
	CommonRes
	Result struct {
		JdOrderState int                 `json:"jdOrderState"` //京东状态
		SubmitState  int                 `json:"submitState"`  //预占确认状态。0没确认预占。   1已确认预占。
		Type         int                 `json:"type"`         //订单类型。1是父订单   2是子订单
		OrderState   int                 `json:"orderState"`   //订单状态。0为取消订单  1为有效
		POrder       OrderDetailPOrder   `json:"pOrder"`
		COrder       []OrderDetailCOrder `json:"cOrder"`
	} `json:"result"`
}

type OrderDetailPOrder struct {
	JdOrderId       int64          `json:"jdOrderId"`       //京东主订单编号
	Freight         float64        `json:"freight"`         //运费
	OrderPrice      float64        `json:"orderPrice"`      //订单总金额
	OrderNakedPrice float64        `json:"orderNakedPrice"` //订单未含税金额
	OrderTaxPrice   float64        `json:"orderTaxPrice"`   //订单税额
	Sku             []OrderSkuItem `json:"sku"`             //商品列表
}

type OrderDetailCOrder struct {
	POrder          int64          `json:"pOrder"` //京东主订单编号
	OrderState      int            `json:"orderState"`
	JdOrderId       int64          `json:"jdOrderId"` //京东子订单编号
	State           int            `json:"state"`
	Freight         float64        `json:"freight"`
	SubmitState     int            `json:"submitState"`
	OrderPrice      float64        `json:"orderPrice"`
	OrderNakedPrice float64        `json:"orderNakedPrice"`
	Type            int            `json:"type"` //商品类型。   0 普通、1 附件、2 赠品、3延保
	OrderTaxPrice   float64        `json:"orderTaxPrice"`
	Sku             []OrderSkuItem `json:"sku"`
}

type OrderDetailChild struct {
	CommonRes
	Result struct {
		JdOrderState int `json:"jdOrderState"` //京东状态
		OrderDetailCOrder
	} `json:"result"`
}

//OrderFreightRes 运费
type OrderFreightRes struct {
	*CommonRes
	Result struct {
		Freight             float64 `json:"freight"`             //总运费
		BaseFreight         float64 `json:"baseFreight"`         //基础运费
		RemoteRegionFreight float64 `json:"remoteRegionFreight"` //偏远地区加收运费
	} `json:"result"`
}

//SkuNums 商品数量
type SkuNums struct {
	SkuID int `json:"skuId"` //商品编号
	Num   int `json:"num"`   //商品数量
}

// Create 下单
func (*orderJD) Create(ctx context.Context, req *OrderCreateReq) (res *OrderCreateRes, err error) {
	method := "order/submitOrder"
	var param = g.Map{
		"thirdOrder":           req.OrderSn,
		"sku":                  gjson.New(req.SkuNums).MustToJsonString(),
		"orderPriceSnap":       gjson.New(req.PriceSnap).MustToJsonString(),
		"name":                 req.Consignee,
		"mobile":               req.Phone,
		"province":             req.Address.Province,
		"city":                 req.Address.City,
		"county":               req.Address.County,
		"town":                 req.Address.Town,
		"address":              req.Address.Desc,
		"email":                server.Email,
		"companyName":          server.CompanyName,
		"submitState":          server.SubmitState,
		"invoicePhone":         server.Mobile,
		"invoiceState":         "2",
		"invoiceType":          req.InvoiceType,
		"selectedInvoiceTitle": "5",
		"invoiceContent":       "1",
		"paymentType":          "4",
		"isUseBalance":         "1",
		"doOrderPriceMode":     "1",
	}
	result, err := server.requestApi(ctx, method, param)
	if err != nil {
		return
	}
	err = gjson.New(result).Scan(&res)
	return
}

// Reflect 反查订单
func (*orderJD) Reflect(ctx context.Context, thirdOrder string) (res *OrderReflectRes, err error) {
	method := "order/selectJdOrderIdByThirdOrder"
	param := g.Map{
		"thirdOrder": thirdOrder,
	}
	result, err := server.requestApi(ctx, method, param)
	if err != nil {
		return
	}
	err = gjson.New(result).Scan(&res)
	return
}

//Detail  订单详情
func (*orderJD) Detail(ctx context.Context, OrderID string) (res *OrderDetailRes, err error) {
	method := "order/selectJdOrder"
	param := g.Map{
		"jdOrderId": OrderID,
		"queryExts": "jdOrderState",
	}

	result, err := server.requestApi(ctx, method, param)
	if err != nil {
		return
	}

	var base *OrderDetailBase
	_ = gjson.New(result).Scan(&base)
	if base.Result.Type == 1 {
		_ = gjson.New(result).Scan(&res)
		return
	} else {
		var child *OrderDetailChild
		_ = gjson.New(result).Scan(&child)
		res = &OrderDetailRes{}
		res.CommonRes = child.CommonRes
		res.Result.Type = child.Result.Type
		res.Result.SubmitState = child.Result.SubmitState
		res.Result.JdOrderState = child.Result.JdOrderState

		res.Result.POrder = OrderDetailPOrder{
			JdOrderId:       child.Result.JdOrderId,
			Freight:         child.Result.Freight,
			OrderPrice:      child.Result.OrderPrice,
			OrderNakedPrice: child.Result.OrderNakedPrice,
			OrderTaxPrice:   child.Result.OrderTaxPrice,
			Sku:             child.Result.Sku,
		}
		child.Result.OrderDetailCOrder.POrder = child.Result.OrderDetailCOrder.JdOrderId
		res.Result.COrder = append(res.Result.COrder, child.Result.OrderDetailCOrder)
	}
	return
}

//Freight  查询运费
func (*orderJD) Freight(ctx context.Context, sku []*SkuNums, province, city, county, town string) (res *OrderFreightRes, err error) {
	method := "order/getFreight"
	param := g.Map{
		"sku":         gjson.New(sku).MustToJsonString(),
		"province":    province,
		"city":        city,
		"county":      county,
		"town":        town,
		"paymentType": "4",
	}
	result, err := server.requestApi(ctx, method, param)
	if err != nil {
		return
	}
	err = gjson.New(result).Scan(&res)
	return
}

// Confirm 确认收货
func (*orderJD) Confirm(ctx context.Context, OrderID string) (res *CommonRes, err error) {
	method := "order/confirmReceived"
	param := g.Map{
		"jdOrderId": OrderID,
	}
	result, err := server.requestApi(ctx, method, param)
	if err != nil {
		return
	}
	err = gjson.New(result).Scan(&res)
	return

}
