package ikc

import (
	"context"
	"github.com/gogf/gf/encoding/gjson"
	"github.com/gogf/gf/frame/g"
)

type orderIkc struct {
}

//Order 订单
var Order = orderIkc{}

type OrderCreateReq struct {
	OrderSn     string            `json:"externalOrderNo"`
	Address     OrderCreateAddr   `json:"address"`
	OuterUserId string            `json:"outerUserId,omitempty"`
	CouponIds   []string          `json:"couponIds,omitempty"`
	List        []OrderCreateItem `json:"productList"`
}

type OrderCreateAddr struct {
	Name     string `json:"name"` //收货人名称
	Mobile   string `json:"mobile"`
	Province string `json:"province"`
	City     string `json:"city"`
	Area     string `json:"area"`
	Street   string `json:"street"`
}

type OrderCreateItem struct {
	LiveId    string `json:"liveId"`          //活动 ID
	ProductId string `json:"productId"`       //商品 ID
	SkuId     string `json:"skuId"`           //SKU ID
	Amount    string `json:"amount"`          //数量
	Price     string `json:"settlementPrice"` //商品结算金额 单位分
}

type OrderCreateRes struct {
	ResultCode    int    `json:"resultCode"`
	ResultMessage string `json:"resultMessage"`
	Data          struct {
		PaymentOrder struct {
			PaymentNo      string `json:"paymentNo"`
			TotalAmount    int    `json:"totalAmount"`
			ShipmentAmount int    `json:"shipmentAmount"`
			DiscountAmount int    `json:"discountAmount"`
			ProductsAmount int    `json:"productsAmount"`
			OrderList      []struct {
				OrderId         string      `json:"orderId"`
				TotalAmount     int         `json:"totalAmount"`
				ShipmentAmount  int         `json:"shipmentAmount"`
				DiscountAmount  int         `json:"discountAmount"`
				OrderStatus     interface{} `json:"orderStatus"`
				ProductsAmount  int         `json:"productsAmount"`
				PaymentStatus   interface{} `json:"paymentStatus"`
				OrderDetailList []struct {
					OrderDetailId     string      `json:"orderDetailId"`
					SkuId             string      `json:"skuId"`
					ProductId         string      `json:"productId"`
					Price             int         `json:"price"`
					Amount            int         `json:"amount"`
					OrderDetailStatus interface{} `json:"orderDetailStatus"`
					AfterSaleStatus   interface{} `json:"afterSaleStatus"`
					ProductStatus     interface{} `json:"productStatus"`
				} `json:"orderDetailList"`
			} `json:"orderList"`
		} `json:"paymentOrder"`
	} `json:"data"`
}

//Create 下单
func (*orderIkc) Create(ctx context.Context, req OrderCreateReq) (res *OrderCreateRes, err error) {
	method := "order/create"
	result, err := post(ctx, method, req)
	_ = gjson.New(result).Scan(&res)
	return
}

//Cancel 支付前整单取消
//`orderId` 订单号
func (*orderIkc) Cancel(ctx context.Context, orderId string) (res *CommonRes, err error) {
	method := "order/before/cancel"
	result, err := post(ctx, method, g.Map{
		"orderId": orderId,
	})
	_ = gjson.New(result).Scan(&res)
	return
}

//Pay 支付
//`orderId` 订单号
func (*orderIkc) Pay(ctx context.Context, orderId string) (res *CommonRes, err error) {
	method := "order/before/cancel"
	result, err := post(ctx, method, g.Map{
		"orderPaymentNo": orderId,
	})
	_ = gjson.New(result).Scan(&res)
	return
}

//PayCancel 支付后整单取消
//`orderId` 订单号
func (*orderIkc) PayCancel(ctx context.Context, orderId string) (res *CommonRes, err error) {
	method := "order/after/cancelByOrder"
	result, err := post(ctx, method, g.Map{
		"orderId": orderId,
	})
	_ = gjson.New(result).Scan(&res)
	return
}

//GoodsCancel 支付后商品取消
//`orderId` 订单号
//`DetailId` 三级单号
func (*orderIkc) GoodsCancel(ctx context.Context, orderId, DetailId string) (res *CommonRes, err error) {
	method := "order/after/cancel"
	result, err := post(ctx, method, g.Map{
		"orderId":       orderId,
		"orderDetailId": DetailId,
	})
	_ = gjson.New(result).Scan(&res)
	return
}

type OrderDetailRes struct {
	ResultCode    int    `json:"resultCode"`
	ResultMessage string `json:"resultMessage"`
	Data          struct {
		Order struct {
			OrderId         string `json:"orderId"`
			LiveId          string `json:"liveId"`
			TotalAmount     int    `json:"totalAmount"`
			ShipmentAmount  int    `json:"shipmentAmount"`
			DiscountAmount  int    `json:"discountAmount"`
			OrderStatus     int    `json:"orderStatus"`
			ProductsAmount  int    `json:"productsAmount"`
			PaymentStatus   int    `json:"paymentStatus"`
			RefundInsurance bool   `json:"refundInsurance"`
			OrderDetailList []struct {
				OrderDetailId     string `json:"orderDetailId"`
				LiveId            string `json:"liveId"`
				SkuId             string `json:"skuId"`
				ProductId         string `json:"productId"`
				Price             int    `json:"price"`
				Amount            int    `json:"amount"`
				OrderDetailStatus int    `json:"orderDetailStatus"`
				AfterSaleStatus   int    `json:"afterSaleStatus"`
				ProductStatus     int    `json:"productStatus"`
				AftersaleDeadline string `json:"aftersaleDeadline"`
			} `json:"orderDetailList"`
		} `json:"order"`
	} `json:"data"`
}

//Detail 详情
//`orderId` 订单号
func (*orderIkc) Detail(ctx context.Context, orderId string) (res *OrderDetailRes, err error) {
	method := "order/queryByOrderId"
	result, err := post(ctx, method, g.Map{
		"orderId": orderId,
	})
	_ = gjson.New(result).Scan(&res)
	return
}

type OrderReflectDetailRes struct {
	ResultCode    int    `json:"resultCode"`
	ResultMessage string `json:"resultMessage"`
	Data          struct {
		PaymentOrder struct {
			PaymentNo       string `json:"paymentNo"`
			TotalAmount     int    `json:"totalAmount"`
			ShipmentAmount  int    `json:"shipmentAmount"`
			DiscountAmount  int    `json:"discountAmount"`
			ProductsAmount  int    `json:"productsAmount"`
			RefundInsurance bool   `json:"refundInsurance"`
			OrderList       []struct {
				OrderId         string `json:"orderId"`
				LiveId          string `json:"liveId"`
				TotalAmount     int    `json:"totalAmount"`
				ShipmentAmount  int    `json:"shipmentAmount"`
				DiscountAmount  int    `json:"discountAmount"`
				OrderStatus     int    `json:"orderStatus"`
				ProductsAmount  int    `json:"productsAmount"`
				PaymentStatus   int    `json:"paymentStatus"`
				OrderDetailList []struct {
					OrderDetailId     string  `json:"orderDetailId"`
					LiveId            string  `json:"liveId"`
					SkuId             string  `json:"skuId"`
					ProductId         string  `json:"productId"`
					Price             int     `json:"price"`
					Amount            int     `json:"amount"`
					DiscountAmount    float64 `json:"discountAmount"`
					SettlementAmount  float64 `json:"settlementAmount"`
					OrderDetailStatus int     `json:"orderDetailStatus"`
					AfterSaleStatus   int     `json:"afterSaleStatus"`
					ProductStatus     int     `json:"productStatus"`
					AftersaleDeadline string  `json:"aftersaleDeadline"`
				} `json:"orderDetailList"`
			} `json:"orderList"`
		} `json:"paymentOrder"`
	} `json:"data"`
}

//ReflectDetail 详情[按外部订单号查询]
func (*orderIkc) ReflectDetail(ctx context.Context, OrderSn string) (res *OrderReflectDetailRes, err error) {
	method := "order/queryByExternalOrderNo"
	result, err := post(ctx, method, g.Map{
		"externalOrderNo": OrderSn,
	})
	_ = gjson.New(result).Scan(&res)
	return
}
