package jingdong

import (
	"context"
	"github.com/gogf/gf/encoding/gjson"
	"github.com/gogf/gf/frame/g"
)

type orderJD struct {
}

//Order 订单
var Order = orderJD{}

//GetDeliveryRes 物流轨迹
type GetDeliveryRes struct {
	*CommonRes
	Result struct {
		OrderTrack []struct {
			Content  string `json:"content"`  //操作内容明细
			MsgTime  string `json:"msgTime"`  //操作时间。日期格式为“yyyy-MM-dd hh:mm:ss”
			Operator string `json:"operator"` //操作员名称
		} `json:"orderTrack"`
		WaybillCode []struct {
			OrderID         string `json:"orderId"`         //订单号
			ParentID        string `json:"parentId"`        //父订单号。	此字段为0 未拆单
			Carrier         string `json:"carrier"`         //承运商
			DeliveryOrderID string `json:"deliveryOrderId"` //运单号
		} `json:"waybillCode"`
		JdOrderID int `json:"jdOrderId"`
	} `json:"result"`
}

//PriceSnap 商品价格
type PriceSnap struct {
	SkuID int `json:"skuId"` //商品编号
	Price int `json:"price"` //商品价格  格式:21.30
}

//PlaceOrderReq 下单
type PlaceOrderReq struct {
	OrderSn   string //订单单号
	SkuNums   []*SkuNums
	Consignee string //收货人姓名，最多20个字符
	Phone     string //手机号，最多20个字符
	Address   *PlaceOrderAddress
	PriceSnap []*PriceSnap
}

//PlaceOrderAddress 下单地址
type PlaceOrderAddress struct {
	Province string //一级地址编码
	City     string //二级地址编码
	County   string //三级地址编码
	Town     string //四级地址编码
	Desc     string //收货人详细地址，最多100个字符
}

//PlaceOrderRes 下单
type PlaceOrderRes struct {
	*CommonRes
	Result struct {
		JdOrderID       int64           `json:"jdOrderId"`       //京东订单号
		OrderPrice      float64         `json:"orderPrice"`      //订单总金额
		OrderNakedPrice float64         `json:"orderNakedPrice"` //订单未税金额
		OrderTaxPrice   float64         `json:"orderTaxPrice"`   //订单税额
		Sku             []*OrderSkuItem `json:"sku"`
	} `json:"result"`
}

//OrderSkuItem 订单商品
type OrderSkuItem struct {
	SkuID      int     `json:"skuId"`      //商品编号
	Num        int     `json:"num"`        //商品数量
	Price      float64 `json:"price"`      //商品单价
	Name       string  `json:"name"`       //商品名称
	NakedPrice float64 `json:"nakedPrice"` //商品未税价
}

//ReflectOrderRes 反查订单
type ReflectOrderRes struct {
	*CommonRes
	Result string `json:"result"` //第三方订单号（非京东订单号）
}

//GetOrderRes 订单详情
type GetOrderRes struct {
	*CommonRes
	Result struct {
		JdOrderID       int64           `json:"jdOrderId"`       //京东订单编号
		Type            int             `json:"type"`            //订单类型。1是父订单   2是子订单
		Freight         float64         `json:"freight"`         //运费
		OrderPrice      float64         `json:"orderPrice"`      //订单总金额
		OrderNakedPrice float64         `json:"orderNakedPrice"` //订单未含税金额
		OrderTaxPrice   float64         `json:"orderTaxPrice"`   //订单税额
		Sku             []*OrderSkuItem `json:"sku"`             //商品列表
		OrderState      int             `json:"orderState"`      //订单状态。0为取消订单  1为有效
		State           int             `json:"state"`           //物流状态。0 是新建  1是妥投   2是拒收
		POrder          struct {
			JdOrderID       int64           `json:"jdOrderId"`       //京东订单编号
			Freight         float64         `json:"freight"`         //运费
			OrderPrice      float64         `json:"orderPrice"`      //订单总金额
			OrderNakedPrice float64         `json:"orderNakedPrice"` //订单未含税金额
			Sku             []*OrderSkuItem `json:"sku"`             //商品列表
		} `json:"pOrder"`
		COrder []struct {
			POrder          string         `json:"pOrder"`          //父订单号。为0时，此订单为父单。
			JdOrderID       int            `json:"jdOrderId"`       //京东订单编号
			Freight         float64        `json:"freight"`         //运费
			OrderPrice      float64        `json:"orderPrice"`      //订单总金额
			OrderNakedPrice float64        `json:"orderNakedPrice"` //订单未含税金额
			Sku             []OrderSkuItem `json:"sku"`             //商品列表
		} `json:"cOrder"`
	} `json:"result"`
}

//GetFreightRes 运费
type GetFreightRes struct {
	*CommonRes
	Result struct {
		Freight             float64 `json:"freight"`             //总运费
		BaseFreight         float64 `json:"baseFreight"`         //基础运费
		RemoteRegionFreight float64 `json:"remoteRegionFreight"` //偏远地区加收运费
	} `json:"result"`
}

//SkuNums 商品数量
type SkuNums struct {
	SkuID int `json:"skuId"` //商品编号
	Num   int `json:"num"`   //商品数量
}

// GetDelivery 物流信息
func (*orderJD) GetDelivery(ctx context.Context, OrderID string) (res *GetDeliveryRes, err error) {
	method := "order/orderTrack"
	param := g.MapStrStr{
		"jdOrderId":   OrderID,
		"waybillCode": "1",
	}
	result, err := server.requestApi(ctx, method, param)
	if err != nil {
		return
	}
	err = gjson.New(result).Scan(&res)
	return
}

// PlaceOrder 下单
func (orderJD) PlaceOrder(ctx context.Context, req *PlaceOrderReq) (res *PlaceOrderRes, err error) {
	method := "order/submitOrder"
	var param = g.MapStrStr{
		"thirdOrder":           req.OrderSn,
		"sku":                  gjson.New(req.SkuNums).MustToJsonString(),
		"orderPriceSnap":       gjson.New(req.PriceSnap).MustToJsonString(),
		"name":                 req.Consignee,
		"mobile":               req.Phone,
		"province":             req.Address.Province,
		"city":                 req.Address.City,
		"county":               req.Address.County,
		"town":                 req.Address.Town,
		"address":              req.Address.Desc,
		"email":                server.Email,
		"companyName":          server.CompanyName,
		"submitState":          server.SubmitState,
		"invoicePhone":         server.Mobile,
		"invoiceState":         "2",
		"invoiceType":          "4",
		"selectedInvoiceTitle": "5",
		"invoiceContent":       "1",
		"paymentType":          "4",
		"isUseBalance":         "1",
		"doOrderPriceMode":     "1",
	}
	result, err := server.requestApi(ctx, method, param)
	if err != nil {
		return
	}
	err = gjson.New(result).Scan(&res)
	return
}

// ReflectOrder 反查订单
func (orderJD) ReflectOrder(ctx context.Context, thirdOrder string) (res *ReflectOrderRes, err error) {
	method := "order/selectJdOrderIdByThirdOrder"
	param := g.MapStrStr{
		"thirdOrder": thirdOrder,
	}
	result, err := server.requestApi(ctx, method, param)
	if err != nil {
		return
	}
	err = gjson.New(result).Scan(&res)
	return
}

// GetOrderInfo 订单详情
func (orderJD) GetOrderInfo(ctx context.Context, OrderID string) (res *GetOrderRes, err error) {
	method := "order/selectJdOrder"
	param := g.MapStrStr{
		"jdOrderId": OrderID,
	}
	result, err := server.requestApi(ctx, method, param)
	if err != nil {
		return
	}
	_ = gjson.New(result).Scan(&res)
	return
}

// GetFreight 查询运费
func (orderJD) GetFreight(ctx context.Context, sku []*SkuNums, province, city, county, town string) (res *GetFreightRes, err error) {
	method := "order/getFreight"
	param := g.MapStrStr{
		"sku":         gjson.New(sku).MustToJsonString(),
		"province":    province,
		"city":        city,
		"county":      county,
		"town":        town,
		"paymentType": "4",
	}
	result, err := server.requestApi(ctx, method, param)
	if err != nil {
		return
	}
	err = gjson.New(result).Scan(&res)
	return
}

// ConfirmReceived 确认收货
func (orderJD) ConfirmReceived(ctx context.Context, OrderID string) (res *CommonRes, err error) {
	method := "order/confirmReceived"
	param := g.MapStrStr{
		"jdOrderId": OrderID,
	}
	result, err := server.requestApi(ctx, method, param)
	if err != nil {
		return
	}
	err = gjson.New(result).Scan(&res)
	return

}
