package tm

import (
	"context"
	"encoding/json"
	"github.com/gogf/gf/container/garray"
	"github.com/gogf/gf/encoding/gjson"
	"github.com/gogf/gf/util/gconv"
)

type orderTm struct {
}

var Order = orderTm{}

type OrderGoodsItem struct {
	ItemId   string `json:"ItemId"`
	Quantity string `json:"Quantity"`
	SkuId    string `json:"SkuId"`
}

type OrderAddress struct {
	AddressDetail string `json:"AddressDetail"`
	DivisionCode  string `json:"DivisionCode"`
	FullName      string `json:"FullName"`
	Mobile        string `json:"Mobile"`
}

type OrderBeforeRes struct {
	Code      string `json:"Code"`
	Message   string `json:"Message"`
	RequestId string `json:"RequestId"`
	Success   bool   `json:"Success"`
	Model     struct {
		RenderOrderInfos struct {
			RenderOrderInfos []struct {
				LmItemInfos struct {
					LmItemInfos []struct {
						ItemId   int64  `json:"ItemId"`
						Quantity int    `json:"Quantity"`
						CanSell  bool   `json:"CanSell"`
						SkuId    int64  `json:"SkuId"`
						Message  string `json:"Message"`
						ItemName string `json:"ItemName"`
					} `json:"LmItemInfos"`
				} `json:"LmItemInfos"`
				DeliveryInfos struct {
					DeliveryInfos []struct {
						ServiceType int    `json:"ServiceType"`
						Id          string `json:"Id"`
						PostFee     int64  `json:"PostFee"`
						DisplayName string `json:"string"`
					} `json:"DeliveryInfos"`
				} `json:"DeliveryInfos"`
			} `json:"RenderOrderInfos"`
		} `json:"RenderOrderInfos"`
	} `json:"Model"`
}

//Before 验证订单商品
func (s *orderTm) Before(ctx context.Context, addr OrderAddress, itemLists []OrderGoodsItem) (res *OrderBeforeRes, err error) {
	method := "renderOrder"

	request := convert("ItemList", gconv.Maps(itemLists))
	request["DeliveryAddress"] = gjson.New(addr).MustToJsonString()
	request["BizUid"] = server.BizUid
	request["AccountType"] = typeEp

	result, err := post(ctx, method, request)
	_ = json.Unmarshal([]byte(result), &res)

	return
}

type OrderCreateReq struct {
	UserId       string
	ChannelOrder string
	Address      OrderAddress
	ItemLists    []OrderGoodsItem
}

type OrderCreateRes struct {
	Code      string `json:"Code"`
	Message   string `json:"Message"`
	RequestId string `json:"RequestId"`
	Success   bool   `json:"Success"`
	Model     struct {
		OrderIds struct {
			OrderIds []string `json:"OrderIds"`
		} `json:"OrderIds"`
		PayTradeIds struct {
			PayTradeIds []string `json:"PayTradeIds"`
		} `json:"PayTradeIds"`
		LmOrderList struct {
			LmOrderList []struct {
				LmOrderId string `json:"LmOrderId"`
			} `json:"LmOrderList"`
		} `json:"LmOrderList"`
	} `json:"Model"`
}

//Create 下单接口
func (s *orderTm) Create(ctx context.Context, req OrderCreateReq) (res *OrderCreateRes, err error) {
	method := "createOrderV2"

	request := convert("ItemList", gconv.Maps(req.ItemLists))
	request["DeliveryAddress"] = gjson.New(req.Address).MustToJsonString()
	request["BizUid"] = server.BizUid
	request["OutTradeId"] = req.ChannelOrder
	request["AccountType"] = typeAnony
	request["ThirdPartyUserId"] = req.UserId

	result, err := post(ctx, method, request)
	_ = json.Unmarshal([]byte(result), &res)
	return
}

type OrderPayReq struct {
	AppID        interface{}
	ChannelOrder string
	LmOrderId    string
}

//Pay 订单支付
func (s *orderTm) Pay(ctx context.Context, req OrderPayReq) (res *OrderCreateRes, err error) {
	method := "enableOrder"
	result, err := post(ctx, method, map[string]string{
		"BizUid":           server.BizUid,
		"LmOrderId":        req.LmOrderId,
		"OutTradeId":       req.ChannelOrder,
		"AccountType":      typeAnony,
		"ThirdPartyUserId": gconv.String(req.AppID),
	})
	_ = json.Unmarshal([]byte(result), &res)
	return
}

type OrderListReq struct {
	UserId          string
	ChannelOrder    string
	LmOrderId       string
	LogisticsStatus string
	OrderStatus     string
}

type OrderFilter struct {
	LmOrderList     []string `json:"lmOrderList,omitempty"`
	ChannelOrder    []string `json:"channelOrderIdList,omitempty"`
	LogisticsStatus string   `json:"LogisticsStatus,omitempty"`
	OrderStatus     string   `json:"OrderStatus,omitempty"`
}

type OrderListRes struct {
	PostFee struct {
		FundAmount      int    `json:"FundAmount"`
		FundType        string `json:"FundType"`
		FundAmountMoney int    `json:"FundAmountMoney"`
	} `json:"PostFee"`
	RequestId   string `json:"RequestId"`
	Message     string `json:"Message"`
	PageSize    int    `json:"PageSize"`
	PageNumber  int    `json:"PageNumber"`
	LmOrderList struct {
		LmOrderList []struct {
			PostFee struct {
				FundAmount      int    `json:"FundAmount"`
				FundType        string `json:"FundType"`
				FundAmountMoney int    `json:"FundAmountMoney"`
			} `json:"PostFee"`
			ExtJson             string `json:"ExtJson"`
			OrderStatus         int    `json:"OrderStatus"`
			ShopName            string `json:"ShopName"`
			EnableStatus        int    `json:"EnableStatus"`
			TbOrderId           int64  `json:"TbOrderId"`
			OrderAmount         int    `json:"OrderAmount"`
			FundStructureModels struct {
				FundStructureModels []struct {
					FundAmount      int    `json:"FundAmount"`
					FundType        string `json:"FundType"`
					FundAmountMoney int    `json:"FundAmountMoney"`
				} `json:"FundStructureModels"`
			} `json:"FundStructureModels"`
			CreateDate      string `json:"CreateDate"`
			LmOrderId       int    `json:"LmOrderId"`
			LogisticsStatus int    `json:"LogisticsStatus"`
			SubOrderList    struct {
				SubOrderList []struct {
					ItemPriceList struct {
						ItemPriceList []struct {
							FundAmount      int    `json:"FundAmount"`
							FundType        string `json:"FundType"`
							FundAmountMoney int    `json:"FundAmountMoney"`
						} `json:"ItemPriceList"`
					} `json:"ItemPriceList"`
					OrderStatus  int    `json:"OrderStatus"`
					Number       int    `json:"Number"`
					EnableStatus int    `json:"EnableStatus"`
					TbOrderId    int64  `json:"TbOrderId"`
					ItemTitle    string `json:"ItemTitle"`
					SkuName      string `json:"SkuName"`
					ItemPic      string `json:"ItemPic"`
					SkuId        string `json:"SkuId"`
					ItemId       int64  `json:"ItemId"`
					LmOrderId    int    `json:"LmOrderId"`
				} `json:"SubOrderList"`
			} `json:"SubOrderList"`
		} `json:"LmOrderList"`
	} `json:"LmOrderList"`
	Code string `json:"Code"`
}

//List 列表
func (s *orderTm) List(ctx context.Context, req OrderListReq) (res *OrderListRes, err error) {
	method := "queryOrderList"

	var filterOption = OrderFilter{
		LogisticsStatus: req.LogisticsStatus,
		OrderStatus:     req.OrderStatus,
	}
	if req.LmOrderId != "" {
		filterOption.LmOrderList = garray.NewStrArray().Append(req.LmOrderId).Slice()
	}
	if req.ChannelOrder != "" {
		filterOption.ChannelOrder = garray.NewStrArray().Append(req.ChannelOrder).Slice()

	}

	request := map[string]string{
		"BizUid":           server.BizUid,
		"FilterOption":     gjson.New(filterOption).MustToJsonString(),
		"PageNumber":       "1",
		"PageSize":         "20",
		"AccountType":      typeAnony,
		"ThirdPartyUserId": req.UserId,
	}

	result, err := post(ctx, method, request)
	_ = json.Unmarshal([]byte(result), &res)
	return
}

type OrderCancelReq struct {
	UserId    string
	LmOrderId string
}

//Cancel 仅适⽤下单[Create]未⽀付[Pay]的订单
func (s *orderTm) Cancel(ctx context.Context, req OrderCancelReq) (res *OrderListRes, err error) {
	method := "cancelOrder"

	result, err := post(ctx, method, map[string]string{
		"BizUid":           server.BizUid,
		"LmOrderId":        req.LmOrderId,
		"AccountType":      typeAnony,
		"ThirdPartyUserId": req.UserId,
	})
	_ = json.Unmarshal([]byte(result), &res)
	return
}
