package jst

import (
	"context"
	"github.com/gogf/gf/encoding/gjson"
	"github.com/gogf/gf/errors/gerror"
	"github.com/gogf/gf/frame/g"
	"github.com/gogf/gf/os/gtime"
	"github.com/gogf/gf/util/gconv"
	"github.com/gogf/gf/util/grand"
	"github.com/gogf/gf/util/gutil"
)

type tokenLogic struct {
}

var Token = tokenLogic{}

type TokenInitReq struct {
	AppKey    string `json:"app_key"`
	Timestamp string `json:"timestamp"`
	GrantType string `json:"grant_type"`
	Charset   string `json:"charset"`
	Code      string `json:"code"`
}

type TokenInitRes struct {
	CommonRes
	Data struct {
		AccessToken  string `json:"access_token"`
		ExpiresIn    int64  `json:"expires_in"`
		RefreshToken string `json:"refresh_token"`
		Scope        string `json:"scope"`
	} `json:"data"`
}

func (s *tokenLogic) Init(ctx context.Context) (res *TokenInitRes, err error) {
	var method = "openWeb/auth/getInitToken"
	var request = TokenInitReq{
		AppKey:    server.AppKey,
		Timestamp: gtime.TimestampStr(),
		GrantType: "authorization_code",
		Charset:   "utf-8",
		Code:      grand.S(6),
	}
	result, err := post(ctx, method, request)
	if err != nil {
		return
	}
	err = gjson.New(result).Scan(&res)
	if err != nil {
		return
	}
	conn := g.Redis().Ctx(ctx).Conn()
	defer func() {
		_ = conn.Close()
	}()
	_, _ = conn.DoVar("SELECT", 10)
	var token = new(TokenCache)
	token.AccessToken = res.Data.AccessToken
	token.RefreshToken = res.Data.RefreshToken
	token.Expires = gtime.Timestamp() + res.Data.ExpiresIn
	_, _ = conn.Do("HMSET", append(g.Slice{CacheKey}, gutil.MapToSlice(gconv.Map(token))...)...)
	return
}

type TokenRefreshReq struct {
	AppKey       string `json:"app_key"`
	Timestamp    string `json:"timestamp"`
	GrantType    string `json:"grant_type"`
	Charset      string `json:"charset"`
	Code         string `json:"code"`
	RefreshToken string `json:"refresh_token"`
	Scope        string `json:"scope"`
}

type TokenRefreshRes struct {
	CommonRes
	Data struct {
		AccessToken  string `json:"access_token"`
		ExpiresIn    int64  `json:"expires_in"`
		RefreshToken string `json:"refresh_token"`
		Scope        string `json:"scope"`
	} `json:"data"`
}

func (s *tokenLogic) Refresh(ctx context.Context, req string) (res *TokenRefreshRes, err error) {
	var method = "openWeb/auth/refreshToken"
	var request = TokenRefreshReq{
		AppKey:       server.AppKey,
		Timestamp:    gtime.TimestampStr(),
		GrantType:    "refresh_token",
		Charset:      "utf-8",
		Code:         grand.S(6),
		RefreshToken: req,
		Scope:        "all",
	}
	result, err := post(ctx, method, request)
	if err != nil {
		return
	}
	err = gjson.New(result).Scan(&res)
	if err != nil {
		return
	}
	if res.Code != 0 {
		err = gerror.New("获取token失败")
		return
	}
	conn := g.Redis().Ctx(ctx).Conn()
	defer func() {
		_ = conn.Close()
	}()
	_, _ = conn.DoVar("SELECT", 10)
	var token = new(TokenCache)
	token.AccessToken = res.Data.AccessToken
	token.RefreshToken = res.Data.RefreshToken
	token.Expires = gtime.Timestamp() + res.Data.ExpiresIn
	_, _ = conn.Do("HMSET", append(g.Slice{CacheKey}, gutil.MapToSlice(gconv.Map(token))...)...)
	return
}

type TokenCache struct {
	AccessToken  string
	RefreshToken string
	Expires      int64
}

func (s *tokenLogic) Get(ctx context.Context) (res string, err error) {
	var token *TokenCache
	conn := g.Redis().Ctx(ctx).Conn()
	defer func() {
		_ = conn.Close()
	}()
	_, _ = conn.DoVar("SELECT", 10)
	cache, _ := conn.DoVar("HGETALL", CacheKey)
	if !cache.IsEmpty() {
		_ = cache.Scan(&token)
		if token.Expires > gtime.Timestamp() {
			res = token.AccessToken
			return
		} else {
			var refresh *TokenRefreshRes
			refresh, err = s.Refresh(ctx, token.RefreshToken)
			if err != nil {
				return
			}
			res = refresh.Data.AccessToken
			return

		}
	}
	var init *TokenInitRes
	init, err = s.Init(ctx)
	if err != nil {
		return
	}
	res = init.Data.AccessToken
	return

}
